
///////////////////////////////////////////////////////////////////////////////
//
//   ROCKWELL SEMICONDUCTOR SYSTEMS - COMMERCIAL GPS BUSINESS
//
///////////////////////////////////////////////////////////////////////////////
//
//
//   ZTYPES.H - Zodiac Message Type Definitions
// 
//
//   DESCRIPTION
//
//   Type definitions for Zodiac messages.
//
//
//   REVISION HISTORY
//
//   $Log:   V:\Projects\Labmon\Include\Archives\ZTYPES.H_v  $
//   
//      Rev 1.18   15 Dec 1999 14:34:42   phungh
//   Add msg structure for 1092, 1292 and take
//   out 1192, 1392 to reflect the msgId changes
//   in the zodiac s/w ver 2.57 and later
//   
//      Rev 1.17   08 Dec 1999 16:36:20   phungh
//   Add new structures for Magna Control, 
//   message 1106, 1191, 1192, 1306, 1334,
//   1337, 1390.  Also, add Build, Decode and 
//   Show functions for these messages
//   
//   
//      Rev 1.16   21 Apr 1999 11:46:18   phungh
//   Labmon54 : Add message definitions for
//   1050 and 1051
//   
//      Rev 1.15   09 Nov 1998 16:58:08   phungh
//   Lbmon511: Add cF7 key combination to
//   the menu to allow choosing antenna type
//   ( passive/active ).  Msg1218 module was
//   added.
//   
//      Rev 1.8   Jul 06 1998 09:13:18   BANHND
//   added more DR features
//   
//      Rev 1.5   Jul 09 1997 09:42:10   COLEJ
//   Multiple Changes...
//   
//      Rev 1.4   Feb 25 1997 14:41:16   COLEJ
//   added msg 1117, 1208, and 1317
//   
//      Rev 1.3   Feb 12 1997 16:04:54   COLEJ
//    
//   
//      Rev 1.2   Nov 14 1996 11:25:02   COLEJ
//    
//   
//      Rev 1.1   Aug 14 1996 10:17:52   COLEJ
//   Added tMSG1108, and MSG1108 to tZMSG.
//   Added show1108 prototype.
//   
//      Rev 1.0   13 May 1996 14:54:36   GPSADMIN
//   Initial release to version control.
//
//
////////////////////////////////////////////////////////////////////////////////
  
#include "gencons.h"


// defines
#define      LEN_BUFFER    1024  // to match msgbuff in labmon.c for now
#define      HDR_LEN       10  

// derived types   
typedef union{
   unsigned short word[LEN_BUFFER/2];
   unsigned char  byte[LEN_BUFFER];
} tMSGBUF;

typedef short     tBIT;        // Derived bit type 

// unpacked differential status bits
//
typedef struct {
   tBOOL StationUnhealthy;
   tBOOL UserDisabled;
} tDIFFSTATUSBITS;
  
// unpacked correction status bits
//
typedef struct {
   tUSHORT PRN;
   tBOOL   EphemerisNotPresent;
   tBOOL   CorrectionsNotPresent;
   tBOOL   UDREHigh;
   tBOOL   SatelliteUnhealthy;
   tBOOL   CorrectionsBad;
   tBOOL   CorrectionsStale;
   tBOOL   IODEMismatch;
} tCORRSTATUSBITS;

// unpacked navigation solution status bits
//
typedef struct {
   tBOOL InvalidAltUsed;          // 1=true
   tBOOL InvalidNoDGPS;           // 1=true
   tBOOL InvalidNumTrk;           // 1=true
   tBOOL InvalidExcEHPE;          // 1=true
   tBOOL InvalidExcEVPE;          // 1=true
   tBOOL InvalidNoDRMeas;         // 1=true
   tBOOL InvalidNoConGPS;         // 1=true
   tBOOL InvalidNoGpsOnly;        // 1=true
   tBOOL TypePropagated;          // 1=propagated
   tBOOL TypeAltUsed;             // 1=altitude used
   tBOOL TypeDGPS;                // 1=DGPS mode
   tBOOL TypePower;               // 1=power off
   tBOOL TypeGPS;                 // 1=GPS only
   tBOOL TypeConGPS;              // 1=GPS and DR
   tBOOL TypeDRCal;               // 1=DR using stored calibration data
   tBOOL TypeZeroSpeed;           // 1=Zero Speed calibration DR
} tSOLUTIONSTATUSBITS;
                   
// unpacked channel summary data bits
//
typedef struct {
   tBOOL MeasUsed;                // 1=used
   tBOOL EphAvail;                // 1=available
   tBOOL DGPSAvail;               // 1=available
   tBOOL MeasValid;               // 1=valid
} tCHNSUMDATA;  

// unpacked initialization validity data bits
//
typedef struct {
   tBOOL TimingMode;              // 1=timing mode used 
   tBOOL SelfSurveyMode;          // 1=self survey mode used
} tINITMODEDATA;   

// unpacked initialization validity data bits
//
typedef struct {
   tBOOL ForceTime;               // 1=force 
   tBOOL GPSTimeValid;            // 1=valid 
   tBOOL UTCTimeValid;            // 1=valid 
   tBOOL LatLonValid;             // 1=valid 
   tBOOL AltitudeValid;           // 1=valid 
   tBOOL SpeedCourseValid;        // 1=valid 
   tBOOL MagneticCourse;          // 1=magnetic 
   tBOOL ClimbRateValid;          // 1=valid 
} tINITVALIDDATA;   

// unpacked restart command word bits
//
//typedef struct {
//   tBOOL InvalidateRAM;       // 1=invalidate RAM
//   tBOOL InvalidateEEPROM;    // 1=invalidate EEPROM    
//   tBOOL InvalidateRTC;       // 1=invalidate RTC
//   tBOOL InvalidateEEPROM_FREQSTD;// 1=invalidate EEPROM except for Freq Std Data
//   tBOOL InvalidateRAM_vEEPROM;   // 1=invalidate RAM except for vEEPROM Data
//   tBOOL ForceColdStart;      // 1=force
//} tCOMMANDWORD;                  

// unpacked restart command word bits
//
typedef struct {
   tBOOL InvalidateRAM;          // 1=invalidate RAM
   tBOOL InvalidateEEPROM;       // 1=invalidate EEPROM    
   tBOOL InvalidateRTC;          // 1=invalidate RTC
   tBOOL InvalidateRAM_vEEPROM;  // 1=invalidate RAM except for vEEPROM Data
                                 // Following fields are used with InvalidateEEPROM 
   tBOOL EEPROMClearOnlyFS;      // 1 = Clear Only EEPROM Freq Std Segments
   tBOOL EEPROMKeepFSCubic;      // 1 = Keep EEPROM Freq Std Cubic
   tBOOL EEPROMKeepFSTable;      // 1 = Keep EEPROM Freq Std Table
   tBOOL EEPROMKeepFSCal;        // 1 = Keep EEPROM Freq Std Calibration
   tBOOL EEPROMKeepAlmanac;      // 1 = Keep EEPROM Almanac
   tBOOL EEPROMKeepDRNav;        // 1 = Keep EEPROM DR Navigation
   tBOOL ForceColdStart;         // 1=force
} tCOMMANDWORD;
   
// unpacked solution validity data bits
//
typedef struct {
   tBOOL AltNotUsedReq;             // 1=required
   tBOOL DGPSRequired;              // 1=required
   tBOOL DRMeasRequired;            // 1=required
   tBOOL ConGPSRequired;            // 1=required
   tBOOL GPSOnlyRequired;           // 1=required
} tSOLVALIDDATA;    
  
// unpacked nav config bits
//
typedef struct {
   tBOOL DisableHeldAltitude;       // 1=disabled
   tBOOL DisableGTSmoothing;        // 1=disabled
   tBOOL DisablePosPinning;         // 1=disabled
   tBOOL DisableLowQualityMeas;     // 1=disabled
   tBOOL EnableJammingDetect;       // 1=enabled
} tNAVCFGBITS;

// unpacked config data bits
//
typedef struct {
   tBOOL PowerMgmtEnable;           // 1=enabled
   tBOOL ColdStartDisable;          // 1=disabled
   tBOOL DGPSDisabled;              // 1=disabled
   tBOOL HeldAltitudeDisabled;      // 1=disabled
   tBOOL GTSmoothingDisabled;       // 1=disabled
   tBOOL PosPinningDisabled;        // 1=disabled
   tBOOL LowQualityMeasDisabled;    // 1=disabled
   tBOOL JammingDetectEnabled;      // 1=enabled
   tBOOL Reserved;
   tBOOL ActiveAntenna;                  // 1=present
   tUSHORT CNThreshold;
} tCONFIGDATA;      

// unpacked serial port validity data bits
//
typedef struct {
   tBOOL Port1;                     // 1=valid
   tBOOL Port2;                     // 1=valid
} tSPVALIDDATA;   

// unpacked serial port validity data bits
//
typedef struct {
   tBOOL Type;                       // 1=active
} tANTENNADATA;  
                
// unpacked DGPS control data bits
//
typedef struct {
   tBOOL DGPSDisable;                // 1=disable
   tBOOL DGPSReset;                  // 1=reset
} tDGPSCONTROLDATA;     
   
// unpacked cold start control data bit
//
typedef struct {
   tBOOL Disable;                    // bit 0
} tCOLDSTARTDATA;

// unpacked user entered altitude input data bits
//
typedef struct {
   tBOOL ForceUse;                   // 1=force
   tBOOL MSLSelect;                  // 1=MSL
   tBOOL StoreRAM;                   // 1=store
   tBOOL StoreEEPROM;                // 1=store
   tBOOL ClearRAM;                   // 1=clear
   tBOOL ClearEEPROM;                // 1=clear
} tUSERALTINPUTDATA;     
        
// unpacked EEPROM update data 
//
typedef struct {
   tUSHORT DataId;                     // 0=STATUS
   tUSHORT PRN;                        // ND
} tEEPROMUPDATEDATA;     
    
// unpacked frequency standard table data 
//
typedef struct {
   tUSHORT Week;                       // 4 * weeks
   tUSHORT Error;                      // 0.15 * ppm
} tFRQTBLDATA;            

//  raw almanac data
//
typedef struct {
   tUSHORT WeekNumber;
   tUSHORT AlmanacData[12];
} tRAWALMDATA;

// unpacked 1170 bits
//
typedef struct {
   tBOOL DRHeadingRateValid;                     // 1=true
   tBOOL DRHeadingRateSensorTempValid;           // 1=true
} tBITS1170;

// unpacked 1171 bits
//
typedef struct {
   tBOOL BackUpStatus;                            // 1=backing
} tBITS1171;

// unpacked BestChanStatus bits
//
typedef struct {
   tBOOL MeasurementUsed;                         // 1=true   bit0
   tBOOL EphemerisAvailable;                      // 1=true   bit1
   tBOOL DifferentialAvailable;                   // 1=true   bit2
   tBOOL MeasurementValid;                        // 1=true   bit3
   tUSHORT CN0;                                   // ND       bits4-9
   tUSHORT PRN;                                   // ND       bits10-15
} tBESTCHANSTATUS;


////////////////////////////////////////////////////////////////////////////////
// MAGNA CONTROL STRUCTURE.
////////////////////////////////////////////////////////////////////////////////
// Used for Input and Output.
typedef struct
{
   tSHORT   SequenceNumber;

   // USER INPUT SECTION.
   // User inputs the Magna Mode to control magna results and power usage.
   // 0 =  Off.               Operate in normal GPS mode.
   // 1 =  Fast Acquisition.  Use Magna to find the satellites.
   // 2 =  On.                Use Magna measurements.
   tSHORT   MagnaMode;

   // 0 = Off. Default = 34 db.
   // A doubling of the search time occurs for every 2.3 dB of lower CNo limit.
   // That means more power and ram space will be used.
   // 1 = 32 db CNo Limit.
   // 2 = 30 db CNo Limit.
   // 3 = 28 db CNo Limit.
   // 4 = 26 db CNo Limit.
   // 5 = 24 db CNo Limit.
   // 6 = 22 db CNo Limit.
   // 7 = 20 db CNo Limit.
   tSHORT   EnableLowCNo;

   tSHORT   dummyin[8];    // Unused User input fields.

   // OUTPUT STATUS SECTION.
   // 0 =  No. Receiver is collecting data and needs full power.
   // 1 =  Yes. User can turn receiver power off.
   tSHORT   OKToPowerDown;

   // 0 =  Off.
   //      Magna Conversion uses magna measurements.
   //      Magna Kernel turns radio power off.
   // !0 = On. Possible values are described by the defines below.
   //      Magna Conversion uses Tacking Loop  measurements.
   //      Magna Kernel leaves radio power on.
   tSHORT   ContinuousTracking;

   // 0  = Not used.
   // !0 = Value in microseconds that the receiver waited until the RTC signal
   //      was captured. When Magna is On and the user is turning power off
   //      after the fix, the user must wake up the receiver before the once
   //      a second RTC signal. The receiver must wait and capture the RTC signal
   //      before starting. If this value is large, the user should wake up the
   //      receiver earlier to reduce the power consumption while the receiver is
   //      waiting for the RTC signal.
   tLONG    RTCInterval;

   tSHORT   dummyout[6];      // Unused Output fields.

} tMagnaControl;

////////////////////////////////////////////////////////////////////////
//
//    ZODIAC OUTPUT (LABMON INPUT) MESSAGE DEFINITIONS
//
////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////
//
//    MSG1000: Geodetic Position Status Message
//
typedef struct {
   tSHORT  SequenceNumber;               // ND
   tSHORT  MeasSequenceNumber;           // ND
   tUSHORT SolutionValidity;             // packed data
   tUSHORT SolutionType;                 // packed data
   tUSHORT NumMeasUsed;                  // SVs
   tUSHORT PolarNav;                     // 1=true
   tUSHORT GPSWeek;                      // weeks
   tULONG  GPSTimeSeconds;               // seconds
   tULONG  GPSTimeNanoseconds;           // nanoseconds
   tUSHORT UTCDay;                       // days
   tUSHORT UTCMonth;                     // months
   tUSHORT UTCYear;                      // years
   tUSHORT UTCHours;                     // hours
   tUSHORT UTCMinutes;                   // minutes
   tUSHORT UTCSeconds;                   // seconds
   tULONG  UTCNanoseconds;               // nanoseconds
   tLONG   Latitude;                     // 1E+08 * radians
   tLONG   Longitude;                    // 1E+08 * radians
   tLONG   Height;                       // 1E+02 * meters
   tSHORT  GeoidalSeparation;            // 1E+02 * meters
   tULONG  GroundSpeed;                  // 1E+02 * meters/sec
   tUSHORT TrueCourse;                   // 1E+03 * radians
   tSHORT  MagneticVariation;            // 1E+04 * radians
   tSHORT  ClimbRate;                    // 1E+02 * meters/sec
   tUSHORT MapDatum;                     // ND
   tULONG  EHPE;                         // 1E+02 * meters
   tULONG  EVPE;                         // 1E+02 * meters
   tULONG  ETE;                          // 1E+02 * meters
   tUSHORT EHVE;                         // 1E+02 * meters/sec
   tLONG   ClockBias;                    // 1E+02 * meters
   tLONG   ClockBiasStdDev;              // 1E+02 * meters
   tLONG   ClockDrift;                   // 1E+02 * meters/sec
   tLONG   ClockDriftStdDev;             // 1E+02 * meters/sec
} tMSG1000;     

////////////////////////////////////////////////////////////////////////
//
//    MSG1001: ECEF Position Status Message
//   ////////  THIS MESSAGE HAS BEEN REDUCED TO ONLY ECEF MESSAGE  //////

typedef struct {
   tSHORT  SequenceNumber;               // ND
   tSHORT  MeasSequenceNumber;           // ND
   tUSHORT SolutionValidity;             // packed data
   tUSHORT SolutionType;                 // packed data
   tUSHORT NumMeasUsed;                  // SVs
   tUSHORT GPSWeek;                      // weeks
   tULONG  GPSTimeSeconds;               // seconds
   tULONG  GPSTimeNanoseconds;           // nanoseconds
   tUSHORT UTCDay;                       // days
   tUSHORT UTCMonth;                     // months
   tUSHORT UTCYear;                      // years
   tUSHORT UTCHours;                     // hours
   tUSHORT UTCMinutes;                   // minutes
   tUSHORT UTCSeconds;                   // seconds
   tULONG  UTCNanoseconds;               // nanoseconds
   tLONG   ECEFPositionX;                // 1E+02 * meters
   tLONG   ECEFPositionY;                // 1E+02 * meters
   tLONG   ECEFPositionZ;                // 1E+02 * meters
   tLONG   ECEFVelocityX;                // 1E+02 * meters/sec
   tLONG   ECEFVelocityY;                // 1E+02 * meters/sec
   tLONG   ECEFVelocityZ;                // 1E+02 * meters/sec
   tUSHORT MapDatum;                     // ND
   tULONG  EHPE;                         // 1E+02 * meters
   tULONG  EVPE;                         // 1E+02 * meters
   tULONG  ETE;                          // 1E+02 * meters
   tUSHORT EHVE;                         // 1E+02 * meters/sec
   tLONG   ClockBias;                    // 1E+02 * meters
   tLONG   ClockBiasStdDev;              // 1E+02 * meters
   tLONG   ClockDrift;                   // 1E+02 * meters/sec
   tLONG   ClockDriftStdDev;             // 1E+02 * meters/sec
} tMSG1001;  

////////////////////////////////////////////////////////////////////////
//
//    MSG1002: Channel Summary Data Message
//
typedef struct {
   tUSHORT ChanStatus;                      // packed data
   tUSHORT PRN;                             // ND
   tUSHORT CN0;                             // dBHz
} tFPCHANSUM;

typedef struct {
   tSHORT      SequenceNumber;              // ND
   tSHORT      MeasSequenceNumber;          // ND
   tUSHORT     GPSWeek;                     // weeks
   tULONG      GPSTimeSeconds;              // seconds
   tULONG      GPSTimeNanoseconds;          // nanoseconds
   tFPCHANSUM  ChanSum[GI_NUM_HW_CHAN];  
} tMSG1002;   

////////////////////////////////////////////////////////////////////////
//
//    MSG1003: Visible Satellites Message
//
typedef struct {
   tUSHORT PRN;                           // ND
   tSHORT Azimuth;                        // 1E+04 * radians
   tSHORT Elevation;                      // 1E+04 * radians
} tFPVISSAT;

typedef struct {
   tSHORT     SequenceNumber;             // ND
   tUSHORT    GDOP;                       // 1E+02 * ND
   tUSHORT    PDOP;                       // 1E+02 * ND
   tUSHORT    HDOP;                       // 1E+02 * ND
   tUSHORT    VDOP;                       // 1E+02 * ND
   tUSHORT    TDOP;                       // 1E+02 * ND
   tUSHORT    NumVisSats;                 // ND
   tFPVISSAT  VisSat[GI_MAX_VIS_SVS];    
} tMSG1003;   

////////////////////////////////////////////////////////////////////////
//
//    MSG1005: Differential GPS Status Message
//
typedef struct {
   tSHORT  SequenceNumber;
   tUSHORT DiffStatus;
   tUSHORT StationID;
   tUSHORT AgeOfLastCorrections;
   tUSHORT NumberObservations;
   tUSHORT CorrStatus[GI_MAX_VIS_SVS];
} tMSG1005;

////////////////////////////////////////////////////////////////////////
//
//    MSG1006: Channel Corrections Message
//
typedef struct {
  tUSHORT  TropoCorr;
  tLONG    IonoCorr;
  tLONG    SVClockCorr;
  tTRIPLE  SVPos[3];
  tSHORT   Azimuth;
  tSHORT   Elevation;
} tFPCHANCOR;

typedef struct {
  tSHORT     SequenceNumber;
  tSHORT     SatMeasSeqNum;
  tFPCHANCOR ChanSum[GI_NUM_HW_CHAN];
} tMSG1006;

////////////////////////////////////////////////////////////////////////
//
//    MSG1007: Channel Measurements Message
//
typedef struct {
  tTRIPLE   PseudoRange;                   // 1E+03 * meters      
  tLONG     PseudoRangeRate;               // 1E+03 * meters/sec    
  tTRIPLE   IntegratedPhase;               // 1E+03 * meters      
  tTRIPLE   IntegPhaseBias;                // 1E+03 * meters      
  tUSHORT   PhaseBiasCount;                // ND         
} tFPCHANMEAS;

typedef struct {
  tSHORT         SequenceNumber;           // ND
  tSHORT         SatMeasSeqNum;            // ND
  tFPCHANMEAS    ChanSum[GI_NUM_HW_CHAN];  // packed data
} tMSG1007;


////////////////////////////////////////////////////////////////////////
//
//    MSG1008: Best Measurement Message
//
typedef struct {
   tBESTCHANSTATUS    BestChanStatus;               // packed data
   tTRIPLEFLX         CodePhase;                    // Pseudo-Range Measurement
   tTRIPLEFLX         CarrierPhase;                 // CCP Measurement
   tLONG              CarrierVelocity;              // Carrier Velocity
   tUSHORT            PhaseBiasCount;               // Phase Bias Count
} tBESTCHANSUM;

typedef struct {
   tSHORT        SequenceNumber;             // ND
   tUSHORT       GPSWeek;                    // weeks
   tULONG        GPSTimeSeconds;             // seconds
   tULONG        GPSTimeNanoseconds;         // nanoseconds
   tUSHORT       NumSatsUsed;                // ND
   tUSHORT       GDOP;                       // 1E+02 * ND
   tUSHORT       PDOP;                       // 1E+02 * ND
   tUSHORT       HDOP;                       // 1E+02 * ND
   tUSHORT       VDOP;                       // 1E+02 * ND
   tUSHORT       TDOP;                       // 1E+02 * ND
   tBESTCHANSUM  BestChanSum[GI_NUM_HW_CHAN];  
   tUSHORT			GPSHeadingError;
   tUSHORT			GPSSpeedError;
   tULONG			GPSPositionError;
   tUSHORT			DRHeadingError;
   tUSHORT			DRSpeedError;
   tULONG			DRPositionError;
} tMSG1008;   


////////////////////////////////////////////////////////////////////////
//
//    MSG1009: ECEF Position Status Message (MINIMUMIZED MESSAGE)
//  

typedef struct {
   tSHORT  SequenceNumber;               // ND
   tSHORT  MeasSequenceNumber;           // ND
   tLONG   ECEFPositionX;                // 1E+02 * meters
   tLONG   ECEFPositionY;                // 1E+02 * meters
   tLONG   ECEFPositionZ;                // 1E+02 * meters
   tLONG   ECEFVelocityX;                // 1E+02 * meters/sec
   tLONG   ECEFVelocityY;                // 1E+02 * meters/sec
   tLONG   ECEFVelocityZ;                // 1E+02 * meters/sec
} tMSG1009;  


////////////////////////////////////////////////////////////////////////
//
//    MSG1010: Map Datum Output Message
//
typedef struct {
   tSHORT   SequenceNumber;                // ND
   tUSHORT  DatumID;                       // ND
   tULONG   SemiMajorAxisInt;              // meters
   tUSHORT  SemiMajorAxisFrac;             // 1E+04 * meters
   tUSHORT  InverseFlatteningInt;          // ND
   tULONG   InverseFlatteningFrac;         // 1E+09 * ND
   tLONG    WGS84OffsetdX;                 // 1E+02 * meters
   tLONG    WGS84OffsetdY;                 // 1E+02 * meters
   tLONG    WGS84OffsetdZ;                 // 1E+02 * meters
} tMSG1010;

////////////////////////////////////////////////////////////////////////
//
//    MSG1011: Receiver ID Message
//
typedef struct {
   tSHORT  SequenceNumber;            
   tCHAR   NumberOfChannels[20];
   tCHAR   SoftwareVersion[20];
   tCHAR   SoftwareDate[20];
   tCHAR   OptionsList[20];
   tCHAR   Reserved[20];
} tMSG1011;

////////////////////////////////////////////////////////////////////////
//
//    MSG1012: User Settings Output Message
//

typedef struct {
   tSHORT  SequenceNumber;          // ND
   tUSHORT Configuration;           // ND
   tUSHORT ColdStartTimeOut;        // seconds
   tUSHORT DGPSCorrTimeOut;         // seconds
   tUSHORT ElevationMask;           // 1E+03 * radians
   tULONG  SelectedCandidates;      // ND
   tUSHORT SolutionValidity;        // ND
   tUSHORT SatsInTrackReq;          // ND
   tULONG  MaxHorizPosError;        // 1E+02 * meters
   tULONG  MaxVertPosError;         // 1E+02 * meters
   tUSHORT ApplicationPlatform;     // 0=default
} tMSG1012;


////////////////////////////////////////////////////////////////////////
//
//    MSG1040: Raw Almanac Message
//
typedef struct {
   tSHORT  SequenceNumber;
   tRAWALMDATA AlmanacData[32];
} tMSG1040;

////////////////////////////////////////////////////////////////////////
//
//    MSG1041: Raw Ephemeris Message
//
typedef struct {
   tSHORT	SequenceNumber;			// sequence number.
   tSHORT	svid;					// Not part of NAV data stream
   tSHORT	momentum_alert_flag;	// Bit 18 of HOW
   tSHORT	synch_antispoof_flag;	// Bit 19 of HOW
//   tLONG	TimeUpdated;			// Time of last update as time since 1/6/80 (sec)
//   tLONG	TimeObsolete;			// Time data is obsolete as time since 1/6/80
//   tLONG	TZeroEphSinceBegin;		// Eph reference time as time since GPS began (s)
//   tSHORT	TZeroEphWeek;			// GPS week of eph reference time (weeks)
//   tLONG	TZeroClkSinceBegin;		// Clk reference time as time since GPS began (s)
//   tSHORT	TZeroClkWeek;			// GPS week of Clk reference time (weeks)
   tUSHORT	raw_data[36];
} tMSG1041;

////////////////////////////////////////////////////////////////////////
//
//    MSG1042: Raw Iono and UTC Corrections Message
//
typedef struct				     // Iono and Utc data
{
   tSHORT SequenceNumber;        // sequence number.
   tSHORT WeekNum;               // full value UTC reference week
   tUSHORT SysDat[12];           // packed iono/utc data words
} tMSG1042;

////////////////////////////////////////////////////////////////////////
//
//	 MSG1050: RAM Status Message
//
typedef struct
{
	tSHORT SeqNum;
	tULONG Failure;
}tMSG1050;

////////////////////////////////////////////////////////////////////////
//
//	 MSG1051: DR System Status Message
//
typedef struct
{
	tSHORT SeqNum;
	tUSHORT GyroFailure;
	tUSHORT DRSpeedFailure;
}tMSG1051;

////////////////////////////////////////////////////////////////////////
//
//	DR Calibration Message Type
//
   
typedef struct
{
	tSHORT SeqNum;
	tUSHORT DataInvalid;    // Bit 0 => Gyro Temperature Invalid if = 1
                            // Bit 1 => Speed scale factor Invalid if = 1
                            // Bit 2 => Heading rate scale factor Invalid if = 1
                            // Bit 3 => Heading rate bias Invalid if = 1
	tSHORT GyroTemp;        // Gyro Temperature
	tSHORT SpeedSF;         // Speed scale factor
	tUSHORT SpeedSF_SD;     // Standard deviation of speed scale factor
	tSHORT HdgRateSF;       // Heading rate scale factor
	tUSHORT HdgRateSF_SD;   // Standard deviation of Heading rate scale factor
	tSHORT HdgRateBias;     // Heading rate bias
	tUSHORT HdgRateBias_SD; // Standard deviation of Heading rate bias
	tULONG DRGPSTow;        // possible use for speed measurement input interval                              
   
} tMSG1070;

////////////////////////////////////////////////////////////////////////
//
//    MSG1071: DR Parameters Output Message
//
typedef struct {
   tSHORT  SequenceNumber;
   tUSHORT DRSpeedSigma;    //standard deviation
   tUSHORT DRSpeedTimeTag;  //resolution
   tUSHORT DRSpeedLatency;
   tUSHORT Reserved[5];
} tMSG1071;

////////////////////////////////////////////////////////////////////////
//
//     Message 1072: Gyro Temperature Table Output Message
//
typedef struct {
   tSHORT  SequenceNumber;             // ND
   tULONG  WriteCounter;
   tSHORT  Speed_SF;
   tUSHORT Speed_SF_Date;
   tSHORT  Speed_SF_Jump;
   tUSHORT HeadingRateBias[64];
   tUSHORT HeadingRateScale[64];
   tUSHORT HeadingRateDate[64];
} tMSG1072;

////////////////////////////////////////////////////////////////////////
//
//     Message 1075: Gyro Temperature factory test result Message
//
typedef struct
{	tSHORT	SequenceNumber;
	tUSHORT	DRFactoryTestResultFlag; // bit 0:  0 = Test valid, 1 = Test invalid;
								     // bit 1:  0 = Gyro sensor valid, 1 = Gyro sensor invalid;
								     // bit 2:  0 = Temp sensor valid, 1 = Temp sensor invalid;
								     // bit 3:  0 = Gyro Bias cal., 1 = Gyro SF cal;
} tMSG1075;

////////////////////////////////////////////////////////////////////////
//
//     Message 1090: 12/07/99 undefined for now
//
typedef struct
{
	tSHORT Null;
} tMSG1090;

////////////////////////////////////////////////////////////////////////
//
//     Message 1091: 12/07/99 undefined for now
//
typedef struct
{
	tSHORT Null;
} tMSG1091;

////////////////////////////////////////////////////////////////////////
//
//    MSG1100: Built-In Test Results Message
//
typedef struct {
   tSHORT  SequenceNumber;       // ND
   tUSHORT ROMFailure;           // 1=failed
   tUSHORT RAMFailure;           // 1=failed
   tUSHORT EEPROMFailure;        // 1=failed
   tUSHORT DPRAMFailure;         // 1=failed
   tUSHORT DSPFailure;           // 1=failed
   tUSHORT RTCFailure;           // 1=failed
   tUSHORT SP1ErrorCount;        // ND
   tUSHORT SP2ErrorCount;        // ND
   tUSHORT SP1ReceiveCount;      // ND
   tUSHORT SP2ReceiveCount;      // ND
   tUSHORT SWVersion;            // ND
} tMSG1100;      

////////////////////////////////////////////////////////////////////////
//
//    MSG1102: Measurement Time Mark Message
// 
typedef struct {
   tSHORT     SequenceNumber;          // Sequence Number for meas. data set
   tLONG      GpsMsrmntTimeIntPrtn;    // GPS Measurement Time Integer Portion 
   tLONG      GpsMsrmntTimeFrctnlPrtn; // GPS Measurement Time Fract. Portion
   tUSHORT    GpsTimeStatus;
   tLONG      MsrmntTimeBaseEpoch;     // Measurment Epoch
   tULONG     SetTime;                 
   tTRIPLEFLX GpsTimePhase;
   tLONG      GpsTimeVelocity;
   tUSHORT    TempMeasurement;         // Counts (0..65535)
   tSHORT     TempRateMeasurement;     // Counts/Min (-32768..32767)
} tMeasurementTimeMark;

typedef struct {
   tUSHORT    FrameIndex;              // Data word frame index
   tUSHORT    ChannelStatus1;          // Status word one
   tUSHORT    ChannelStatus2;          // Status word two
   tSHORT     PRN;                     // Satellite PRN
   tSHORT     CN0;                     // Signal to Noise Ratio
   tTRIPLEFLX CodePhase;               // Pseudo-Range Measurement
   tTRIPLEFLX CarrierPhase;            // CCP Measurement
   tLONG      CarrierRate;             // Carrier Velocity (DR) Measurement
   tSHORT     CodePhaseError;          // Code Phase Std Deviation
   tSHORT     CarrierPhaseError;       // Carrier Phase Std Deviation
   tULONG     DataWord[2];             // Data word 1 and 2
} tChannelOutput;

typedef tChannelOutput  tChannelOutputArray[GI_NUM_HW_CHAN]; 

typedef struct {
   tMeasurementTimeMark TimeMarkData;
   tChannelOutputArray  ChannelData;       
} tMSG1102;

////////////////////////////////////////////////////////////////////////
//
//    MSG1106: Explicit Acknowledge message
//
typedef struct {
	    tSHORT  SeqNo;
		tSHORT  MsgID;
		tUSHORT AckStatus;  		// bit map : bit0=1(ACK), bit1=1(NAK), bit2-15 reserved
		tUSHORT vEEPBlockBitMap;    // bit map : received status of vEEP block report by Rx.
									// bit0=received status of vEEP first block, and etc...
		tSHORT  Word2;      		// reserved
}tMSG1106;

////////////////////////////////////////////////////////////////////////
//
//    MSG1108: UTC Timemark Pulse Output Message
//
typedef struct {
   tSHORT  SequenceNumber;   // word 8
   tUSHORT UTCDay;           // word 9 - 13 Reserved
   tUSHORT UTCMonth;
   tUSHORT UTCYear;
   tUSHORT UTCHours;
   tUSHORT UTCMinutes;       // word 13
   tULONG  UTCSeconds;       // words 14 - 15
   tUSHORT GPS2UTCIOS;       // word 16  GPS to UTS time offset (integer part)
   tLONG   GPS2UTCFOS;       // words 17 - 18 GPS to UTC time offset (fractional part in nanosecs)
   tUSHORT UTCTimeValidity;  // word 19 (bit0 = Time Mark Validity  bit1 = GPS/UTC Synch bits2-15 Reserved)
   tUSHORT Reserved;         // word 20 Reserved
} tMSG1108;

////////////////////////////////////////////////////////////////////////
//   
//    MSG1110: Frequency Standard Input Parameters In Use Message
//
typedef struct {
  tSHORT         SequenceNumber; // ND  
  tUSHORT        FSIssue;        // ND                                  
  struct {                       
    tSHORT       C0;             // 2^29 * sec/sec
    tSHORT       C1;             // 2^35 * sec/sec/deg C
    tSHORT       C2;             // 2^41 * sec/sec/deg C^2
    tSHORT       C3;             // 2^47 * sec/sec/deg C^3
    tSHORT       TInf;           // 100 * deg C
  } Characteristics; 
  
  struct {
    tSHORT       D0;             // ND
    tSHORT       D1;             // ND
  } Dynamics; 
  
  struct {
    tSHORT       TRef;           // 100 * deg C
    tUSHORT      T0;             // counts
    tSHORT       S0;             // 2^18 * deg C/count
  } Calibration;    
  
  struct {
    tSHORT       U0;             // 2^29 * sec/sec
    tSHORT       U1;             // 2^35 * sec/sec/deg C
  } Uncertainty;
} tMSG1110;

////////////////////////////////////////////////////////////////////////
//   
//    MSG1117: Power Management Duty Cycle In Use Message
//
typedef struct {
  tSHORT         SequenceNumber; // ND  
  tUSHORT        DutyCycle;      // ND                                  
} tMSG1117;

///////////////////////////////////////////////////////////////////////
//
// Message 1130: Serial Port Comm Parameters In Use Message
//
typedef struct {
   tUSHORT DataBits;
   tUSHORT StopBits;
   tUSHORT Parity;
   tUSHORT BitRate;
   tUSHORT Prescale;
   tUSHORT Postscale;
} tSPCOMOUT;

typedef struct {
   tSHORT    SequenceNumber;       // ND
   tSPCOMOUT Port1;
   tSPCOMOUT Port2;
} tMSG1130;

////////////////////////////////////////////////////////////////////////
//
//    Message 1135: EEPROM Update Message
//
typedef struct {
   tSHORT SequenceNumber;
} tMSG1135;      

////////////////////////////////////////////////////////////////////////
//
//     Message 1136: EEPROM Status Message
//
typedef struct {
   tSHORT  SequenceNumber;
   tUSHORT Device;
   tULONG  AlmFailure;
   tULONG  Failure;
   tULONG  AlmStatus;
   tULONG  Status;
} tMSG1136;  


///////////////////////////////////////////////////////////////////////
//
// Message 1137: EEPROM Dump
//
typedef struct {
   tSHORT  SequenceNumber;
   tUSHORT BlockID;
   tUSHORT BlockTotal;
   tUSHORT Data[128];
} tMSG1137;

////////////////////////////////////////////////////////////////////////
//
//     Message 1150: Frequency Standard Table Output  Message
//
typedef struct {
	tSHORT  SequenceNumber;
	tUSHORT Preamble;
	tUSHORT Type;
	tUSHORT StationID;
	tUSHORT ModZCount;
	tUSHORT RTCMSeqNum;
	tUSHORT FrameLength;
	tUSHORT StationHealth;
	tUSHORT NumOfObserve;
} tMSG1150;

////////////////////////////////////////////////////////////////////////
//
//     Message 1160: Frequency Standard Table Output  Message
//
typedef struct {
   tSHORT SequenceNumber;             // ND
   tSHORT TableFreqOffset;             // .15 * ppm
   tSHORT TableFreqOffsetValid;        // 1=valid
   tSHORT OffsetErrorEstimate;         // .002 * ppm
   tSHORT AgingRateEstimate;           // .0002 * ppm/yr
   tSHORT LastRateUpdateWeek;          // week
   tUSHORT FreqStdTable[256];                               
} tMSG1160;

////////////////////////////////////////////////////////////////////////
//
//     Message 1170: DR Heading Rate and Sensor Temperature Measurement Output  Message
//
typedef struct {
   tSHORT  SequenceNumber;              // ND
   tUSHORT Bits1170;                    // packed bitfield
   tSHORT  DRHeadingRate;               // e-02
   tSHORT  DRHeadingRateSensorTemp;     // e-02
} tMSG1170;

////////////////////////////////////////////////////////////////////////
//
//     Message 1171: Time Tagged DR Speed Measurement Output  Message
//
typedef struct {
   tSHORT  SequenceNumber;             // ND
   tSHORT  DRSpeed;                    // E-02
   tUSHORT DRSpeedTimeTag;             // ND
   tUSHORT Bits1171;                   // packed bitfield
} tMSG1171;

////////////////////////////////////////////////////////////////////////
//
//     Message 1172: DR Heading Rate and Sensor Temperature Measurement Output  Message
//
typedef struct {
   tSHORT  SequenceNumber;              // ND
   tUSHORT Bits1170;                    // packed bitfield
   tSHORT  DRHeadingRate;               // e-02
   tSHORT  DRHeadingRateSensorTemp;     // e-02
} tMSG1172;

////////////////////////////////////////////////////////////////////////
//
//     Message 1173: GPS Time tagged DR Measurement Message
//
typedef struct DRMeasDef1173
{
   tLONG MeasTime;       // Time of measurement in estimated GPS time (sec) LSB=0.01
   tLONG HeadingRate;    // degrees/second LSB=0.01
   tLONG HeadingRateCount;    // degrees/second
   tLONG GyroTemp;       // degrees C LSB=0.01
   tLONG Speed;          // meters/second LSB=0.01
   tUSHORT WheelTickCount;
   tUSHORT DRMeasValid;	// bit0: 1=Gyro HR valid; 0 invalid
   						// bit1: 1=Gyro Temp valid; 0 invalid
   						// bit2: 1=backing up; 0=going forward
   tUSHORT Reserved[3];
}  tDRMEAS1173;

// when this is changed make sure that neoutput.h got changed as well.
typedef struct       // structure to hold one second of data
{
      tSHORT SeqNo;
      tDRMEAS1173 DrMeas[10];
   
} tMSG1173;

////////////////////////////////////////////////////////////////////////
//
//     Message 1180: Boot Status Output  Message (only used for flash reprogramming)
//
typedef struct {
   tUSHORT StatusNumber;               // ND
} tMSG1180;


////////////////////////////////////////////////////////////////////////
//
//     Message 1190: Error and Status Message
//
typedef struct {
   tSHORT  SequenceNumber;         // ND
   tUSHORT Class;                  // 0=user mode exception
   tSHORT  ErrorNumber;            // ND
   tUSHORT CodeEnviron;            // ND
   tUSHORT ProgramCounter;         // ND
} tMSG1190; 


#define MX_SATS_OUT (12)      // Maximum number of satellite data buffers in tMagnaMeas

typedef struct 
{
	tSHORT SatPRN;              // Satellite PN code Number [1..32] - Just GPS for now
    tSHORT Doppler;             // Estimated Doppler for this Sat - [-32768..32767] 0.2 Hz
    tUSHORT DopplerUncert;      // Estimated Doppler Uncertainty - [0..65535] 0.1 Hz
                                // Interpret uncertainty 0 as flag for "unknown" Doppler
                                // Single sided: apply same value to both sides
    tULONG CodePhase;           // Estimated Code Phase mod 1 ms - [0..1022999] 0.001 C/A chip
    tUSHORT CodePhaseUncert;    // Estimated Code Phase Uncertainty - [0..10000] 0.001 C/A chips
                                // Interpret uncertainty 0 as flag for "unknown" code phase
                                // Single sided: apply same value to both sides
    tUSHORT SNR;                // Ratio (Peak-MeanNoise)/(RmsAboutMeanNoise) - [0..65535]
    tSHORT CN0;                 // Carrier to Noise Density - 0.1 dB-Hz
}  tSatMeas;

/*
typedef struct 
{
   tSHORT SequenceNumber;
   tUSHORT ValidityFlags;            // 1=Valid, 0=Invalid                                       // Bit 0: Doppler parameters are valid
                                     // Bit 1: Code Phase & SNR parameters are valid
                                     // Bit 2: GPS XO parameters are valid
                                     // Bit 3: GPS Reference Time is valid
                                     // Bit 4: Command is complete
                                     // Bit 5: Failed: Magna not responding
                                     // Bit 6: Abort: Command did not complete normally
                                     // Bit 7: Continuous Mode is valid
   tULONG GpsRefTimeInt;             // GPS Reference Time   - Integer [0..604799]   sec
   tULONG GpsRefTimeFrac;            // GPS Reference Time - fractional [0..999999999] nsec
                                     // Ref Time is GPS time at start of data capture.
   tULONG MeasT20;                   // Set time (T10 count) of T20 following start of data capture
   tULONG MeasOffset;                // 44 MHz count from start of data capture to T20 from Magna
   tSHORT XOError;                   // GPS XO Frequency Error [-32768..32767] 0.01 ppm
                                     // Estimated from min/max observed satellite Dopplers
   tUSHORT XOErrorUncert;            // GPS XO Frequency Error Uncertainty  [0..65535] 0.01 ppm
                                     // Single sided uncertainty (apply same value to both sides)
   tSHORT NumVisSats;                // Number of visible satellites [0..12]
   tSatMeas SatMeas[MX_SATS_OUT];    // Data for each satellite
} tMSG1191; */    

   typedef struct 
   {
      tSHORT SequenceNumber;
      tUSHORT ValidityFlags;            // 1=Valid, 0=Invalid
                                            // Bit 0: Doppler parameters are valid
                                            // Bit 1: Code Phase & SNR parameters are valid
                                            // Bit 2: GPS XO parameters are valid
                                            // Bit 3: GPS Reference Time is valid
                                            // Bit 4: Command is complete
                                            // Bit 5: Failed: Magna not responding
                                            // Bit 6: Abort: Command did not complete normally
                                            // Bit 7: Continuous Mode is valid
                                        // GPS Time at MeasT20
      tULONG GpsRefTimeInt;             // GPS Reference Time   - Integer [0..604799]   sec
      tULONG GpsRefTimeFrac;            // GPS Reference Time - fractional [0..999999999] nsec
                                        // Ref Time is GPS time at start of data capture.
      tULONG MeasT20;                   // Set time (T10 count) of T20 following start of data capture
      tULONG MeasOffset;                // 44 MHz count from start of data capture to T20 from Magna
      tSHORT XOError;                   // GPS XO Frequency Error [-32768..32767] 0.01 ppm
                                        // Estimated from min/max observed satellite Dopplers
      tUSHORT XOErrorUncert;            // GPS XO Frequency Error Uncertainty  [0..65535] 0.01 ppm
                                        // Single sided uncertainty (apply same value to both sides)
      tSHORT NumVisSats;                // Number of visible satellites [0..12]

      // Reference data from last ME Measurement Time Mark Record

      tLONG      MEGpsMsrmntTimeIntPrtn;      // ME GPS Measurement Time Integer Portion
      tLONG      MEGpsMsrmntTimeFrctnlPrtn;   // ME GPS Measurement Time Fract. Portion
      tUSHORT    MEGpsTimeStatus;             // (&0x0004)? GPSTime Set : GPSTime NOT set
                                              // (&0x0020)? ME Clock Repartition : No ME Clock Repartition
                                              // (&0x0040)? RF Power ON : RF Power OFF
                                              // (&0x0080)? Measure slew applied : No measure slew applied (1 hz)
      tULONG     MESetTime;                   // ME Reference Set Time for time bias and drift
      tTRIPLEFLX MEGpsTimePhase;              // ME Time Bias
      tLONG      MEGpsTimeVelocity;           // ME Drift

      tSatMeas SatMeas[MX_SATS_OUT];    // Data for each satellite
   }  tMSG1191;

//////////************************************************//////////////
// NOTE : Message 1192 is changed to 1092 for Zodiac Software version
//        2.57 and later.  Labmon is changed to match this new ID starting
//        from ver 5.9
//////////************************************************//////////////
////////////////////////////////////////////////////////////////////////
//
//     Message 1192: Magna Control Output Message
//
//typedef tMagnaControl tMSG1192;

////////////////////////////////////////////////////////////////////////
//
//     Message 1092: Magna Control Output Message
//
typedef tMagnaControl tMSG1092;


////////////////////////////////////////////////////////////////////////
//
//    ZODIAC INPUT (LABMON OUTPUT) MESSAGE DEFINITIONS
//
////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////
//
//    MSG1200: Geodetic Position and Velocity Initialization Message
//
typedef struct {
   tSHORT  SequenceNumber;     // ND
   tUSHORT Validity      ;     // packed initialization validity bits
   tUSHORT GPSWeek;            // weeks
   tULONG  GPSTimeSeconds;     // seconds
   tUSHORT UTCDay;             // days
   tUSHORT UTCMonth;           // months
   tUSHORT UTCYear;            // years
   tUSHORT UTCHours;           // hours
   tUSHORT UTCMinutes;         // minutes
   tUSHORT UTCSeconds;         // seconds
   tLONG   Latitude;           // 1E+09 * radians 
   tLONG   Longitude;          // 1E+09 * radians
   tLONG   Altitude;           // 1E+02 * meters
   tULONG  GroundSpeed;        // 1E+02 * meters/sec
   tUSHORT Course;             // 1E+03 * radians
   tSHORT  ClimbRate;          // 1E+02 * meters/sec
} tMSG1200;  

////////////////////////////////////////////////////////////////////////
//
//    MSG1208: UTC Time Mark Initialization Message
//
typedef struct {
   tSHORT  SequenceNumber ;    // ND
   tUSHORT Initmode       ;    // packed initialization bits
   tULONG  CableDelay     ;    // nanoseconds delay in cable
   tLONG   SurveyPositionX;    // 1E+02 * meters
   tLONG   SurveyPositionY;    // 1E+02 * meters
   tLONG   SurveyPositionZ;    // 1E+02 * meters
   tUSHORT Reserved;          
} tMSG1208;  

////////////////////////////////////////////////////////////////////////
//
//    MSG1210: Map Datum Output Message
//
typedef struct {
   tSHORT  SequenceNumber;             // ND
   tUSHORT DatumID;                    // ND
   tULONG  SemiMajorAxisInt;           // meters
   tUSHORT SemiMajorAxisFrac;          // 1E+04 * meters
   tUSHORT InverseFlatteningInt;       // ND
   tULONG  InverseFlatteningFrac;      // 1E+09 * ND
   tLONG   WGS84OffsetdX;              // 1E+02 * meters
   tLONG   WGS84OffsetdY;              // 1E+02 * meters
   tLONG   WGS84OffsetdZ;              // 1E+02 * meters
} tMSG1210;

////////////////////////////////////////////////////////////////////////
//
//    Message 1211: Map Datum Select
//
typedef struct {
   tSHORT  SequenceNumber;              // ND
   tUSHORT DatumID;                     // ND
} tMSG1211;

////////////////////////////////////////////////////////////////////////
//
//    MSG1212: Satellite Elevation Mask Control Message
//
typedef struct {
   tSHORT  SequenceNumber;             // ND
   tUSHORT ElevationMask;              // 1E+03 * radians
} tMSG1212;

////////////////////////////////////////////////////////////////////////
//
// Message 1213: Satellite Candidate Select
//
typedef struct {
   tSHORT  SequenceNumber;             // ND
   tULONG  SelectedCandidates;         // packed data
   tUSHORT StorageSelect;
} tMSG1213;

////////////////////////////////////////////////////////////////////////
//
//    Message 1214: Differential GPS Control Message
//
typedef struct {
   tSHORT  SequenceNumber;             // ND
   tUSHORT Control;                    // ND
   tUSHORT TimeOut;                    // secs
} tMSG1214;

////////////////////////////////////////////////////////////////////////
// 

////////////////////////////////////////////////////////////////////////
//   
//   Message 1216: Cold Start Control
//
typedef struct {
   tSHORT  SequenceNumber;
   tUSHORT Control;
   tUSHORT TimeOut;
} tMSG1216;

////////////////////////////////////////////////////////////////////////
//  
//    Message 1217: Solution Validity Criteria
//
typedef struct {
   tSHORT  SequenceNumber;
   tUSHORT Flags;
   tUSHORT MinimumSats;
   tULONG  MaxEHPE;
   tULONG  MaxEVPE;
} tMSG1217;

//   Not Implemented
////////////////////////////////////////////////////////////////////////
//  
// Message 1218: Antenna Type Select
//
typedef struct {
   tSHORT  SequenceNumber;
   tUSHORT Type;
} tMSG1218;


////////////////////////////////////////////////////////////////////////
// 
// Message 1219: User-Entered Altitude
//
typedef struct {
   tSHORT  SequenceNumber;             // ND
   tUSHORT Flags;                      // packed data
   tLONG   Altitude;                   // 1E+02 * meters
   tULONG  AltStdDev;                  // 1E+02 * meters
} tMSG1219;

////////////////////////////////////////////////////////////////////////
//  
// Message 1220: Application Platform Control
//
typedef struct {
   tSHORT  SequenceNumber;              // ND
   tUSHORT Platform;                    // 0=default
} tMSG1220;

////////////////////////////////////////////////////////////////////////
//  
// Message 1221: Navigation Configuration
//
typedef struct {
   tSHORT      SequenceNumber;
   tUSHORT     NavCfgWord;
   tUSHORT     Authorization;
   tUSHORT     PropNavCfgWord;
   tUSHORT     CN0Threshold;
   tUSHORT     Reserved[4];
} tMSG1221;

////////////////////////////////////////////////////////////////////////
//
//    MSG1240: Raw Almanac Input Message
//
typedef struct {
   tSHORT  SequenceNumber;
   tRAWALMDATA AlmanacData[32];
} tMSG1240;

////////////////////////////////////////////////////////////////////////
//
//    MSG1241: Raw Ephemeris Message
//
typedef struct {
   tSHORT	SequenceNumber;			// sequence number.
   tSHORT	svid;					// Not part of NAV data stream
   tSHORT	momentum_alert_flag;	// Bit 18 of HOW
   tSHORT	synch_antispoof_flag;	// Bit 19 of HOW
//   tLONG	TimeUpdated;			// Time of last update as time since 1/6/80 (sec)
//   tLONG	TimeObsolete;			// Time data is obsolete as time since 1/6/80
//   tLONG	TZeroEphSinceBegin;		// Eph reference time as time since GPS began (s)
//   tSHORT	TZeroEphWeek;			// GPS week of eph reference time (weeks)
//   tLONG	TZeroClkSinceBegin;		// Clk reference time as time since GPS began (s)
//   tSHORT	TZeroClkWeek;			// GPS week of Clk reference time (weeks)
   tUSHORT	raw_data[36];
} tMSG1241;

////////////////////////////////////////////////////////////////////////
//
//    MSG1242: Raw Iono and UTC Corrections Message
//
typedef struct				     // Iono and Utc data
{
   tSHORT SequenceNumber;        // sequence number.
   tSHORT WeekNum;               // full value UTC reference week
   tUSHORT SysDat[12];           // packed iono/utc data words
} tMSG1242;

////////////////////////////////////////////////////////////////////////
//
//    MSG1270: DR Initialization Input Message
//
typedef struct {
   tSHORT  SequenceNumber;
   tUSHORT DRSpeedSigma;    //standard deviation
   tUSHORT DRSpeedTimeTag;  //resolution
   tUSHORT DRSpeedLatency;
   tUSHORT Reserved[5];
} tMSG1270;

////////////////////////////////////////////////////////////////////////
//
//    MSG1300: Perform Built-In Test Command Message
//
typedef struct {
   tSHORT         SequenceNumber;      // ND
   tUSHORT        ReservedWord;        // ND
} tMSG1300; 

////////////////////////////////////////////////////////////////////////
//
//    MSG1303: Restart Message
//
typedef struct {
   tSHORT         SequenceNumber;      // ND
   tUSHORT        Command;             // ND
} tMSG1303; 

////////////////////////////////////////////////////////////////////////
//
//    MSG1304: Factory Test Message
//
typedef struct {
   tSHORT         SequenceNumber;      // ND
} tMSG1304; 

////////////////////////////////////////////////////////////////////////
//
//    MSG1305: Gyro Factory Test Message
//
typedef struct {
   tSHORT	SequenceNumber;      // ND
   tUSHORT	DRFactoryTestFlag;
} tMSG1305; 


////////////////////////////////////////////////////////////////////////
//
//    MSG1306: Explicit Acknowledge Input Message
//
typedef struct {
   tSHORT	SequenceNumber;      
   tUSHORT	MsgID;
   tUSHORT  AckStatus;
   tUSHORT  Reserved1;
   tUSHORT  Reserved2;
} tMSG1306; 

////////////////////////////////////////////////////////////////////////
//   
//    MSG1310: Frequency Standard Input Parameters Message
//
typedef struct {
  tSHORT         SequenceNumber; // ND  
  tUSHORT        FSIssue;        // ND                                  
  struct {                       
    tSHORT       C0;             // 2^29 * sec/sec
    tSHORT       C1;             // 2^35 * sec/sec/deg C
    tSHORT       C2;             // 2^41 * sec/sec/deg C^2
    tSHORT       C3;             // 2^47 * sec/sec/deg C^3
    tSHORT       TInf;           // 100 * deg C
  } Characteristics; 
  
  struct {
    tSHORT       D0;             // ND
    tSHORT       D1;             // ND
  } Dynamics; 
  
  struct {
    tSHORT       TRef;           // 100 * deg C
    tUSHORT      T0;             // counts
    tSHORT       S0;             // 2^18 * deg C/count
  } Calibration;    
  
  struct {
    tSHORT       U0;             // 2^29 * sec/sec
    tSHORT       U1;             // 2^35 * sec/sec/deg C
  } Uncertainty;
} tMSG1310;

////////////////////////////////////////////////////////////////////////
//   
//    MSG1317: Power Management Control Message
//
typedef struct {
  tSHORT         SequenceNumber; // ND  
  tUSHORT        DutyCycle;      // ND                                  
} tMSG1317;

////////////////////////////////////////////////////////////////////////
//
// Message 1330: Serial Port Communication Parameters Message
//
typedef struct {
   tUSHORT DataBits;                   // 0=7 bits
   tUSHORT StopBits;                   // 0=1 bit
   tUSHORT Parity;                     // 0=none
   tUSHORT BitRate;                    // 0=custom
   tUSHORT Prescale;                   // ND
   tUSHORT Postscale;                  // ND
} tSPCOMIN;

typedef struct {
   tSHORT   SequenceNumber;            // ND
   tUSHORT  Validity;                  // 1=valid
   tSPCOMIN Port1;                      
   tSPCOMIN Port2;                    
} tMSG1330;

////////////////////////////////////////////////////////////////////////
//
// Message 1331: Message Protocol Control Message
//
typedef struct {
   tSHORT  SequenceNumber;
   tUSHORT Stream;
   tUSHORT Protocol;
} tMSG1331;

////////////////////////////////////////////////////////////////////////
//
// Message 1334: vEEPROM/EEPROM Availability Message
//
typedef struct {
   tSHORT  SequenceNumber;
   tUSHORT Status;
   tUSHORT Reserved1;
   tUSHORT Reserved2;
} tMSG1334;   


////////////////////////////////////////////////////////////////////////
//
// Message 1337: vEEPROM/EEPROM Block Input Message
//
typedef struct {
   tSHORT  SequenceNumber;
   tUSHORT BlockID;
   tUSHORT BlockTotal;
   tUSHORT DataWord[128];
} tMSG1337;   


////////////////////////////////////////////////////////////////////////
//
//     Message 1360: Frequency Standard Table Input Message
//
typedef struct {
   tSHORT SequenceNumber;             // ND
   tSHORT TableFreqOffset;            // .15 * ppm
   tSHORT TableFreqOffsetValid;       // 1=valid
   tSHORT OffsetErrorEstimate;        // .002 * ppm
   tSHORT AgingRateEstimate;          // .0002 * ppm/yr
   tSHORT LastRateUpdateWeek;         // week
   tUSHORT FreqStdTable[256];
} tMSG1360;

////////////////////////////////////////////////////////////////////////
//
//     Message 1380: Flash Reprogram Request Message (only used for flash reprogramming)
//
typedef struct {
   tBOOL          RequestFlag;         // ND
} tMSG1380;


#define MX_SATS_IN (12)    		// Maximum number of satellite data buffers in tMagnaCmd

typedef struct {
	tSHORT SatPRN;				// Satellite PN code Number [1..32] - Just GPS for now
	tSHORT Doppler;				// Estimated Doppler for this Sat - [-32768..32767] 0.2 Hz
	tUSHORT DopplerUncert;		// Estimated Doppler Uncertainty - [0..65535] 0.1 Hz
		                        // This is a single sided value around Doppler
	tUSHORT CodePhase;			// Estimated Code Phase mod 1 ms - [0..1022] 1 C/A chip
	tUSHORT CodePhaseUncert;	// Estimated Code Phase Uncertainty - [0..1023] C/A chips
	                      		// This is a single sided uncertainty around CodePhase
}	tSatData;

typedef struct {
	tSHORT SequenceNumber;
	tUSHORT ValidityFlags;			// 1=Valid, 0=Invalid
									// Bit 0: Doppler parameters are valid
									// Bit 1: Code Phase parameters are valid
									// Bit 2: GPS XO parameters are valid
									// Bit 3: GPS Reference Time is valid
									// Bit 4: Acquire new receiver samples
									// Bit 5: Enable iterated (low C/No) search
									// Bit 6: Continuous Mode is valid
	tULONG GpsRefTimeInt;			// GPS Reference Time	- Integer [0..604799]	sec
	tULONG GpsRefTimeFrac;			// GPS Reference Time - fractional [0..999999999] nsec
	tSHORT XOError;					// GPS XO Frequency Error [-32768..32767] 0.01 ppm
	tUSHORT XOErrorUncert;			// GPS XO Frequency Error Uncertainty	[0..65535] 0.01 ppm
		                            // This is a single-sided value around XOError
	tSHORT NumVisSats;				// Number of visible satellites [0..12]
	tSatData SatData[MX_SATS_IN];	// Data for each satellite
}	tMSG1390;

//////////************************************************//////////////
// NOTE : Message 1392 is changed to 1292 for Zodiac Software version
//        2.57 and later.  Labmon is changed to match this new ID starting
//        from ver 5.9
//////////************************************************//////////////

////////////////////////////////////////////////////////////////////////
//
//     Message 1392: Magna Control Input Message
//
//typedef tMagnaControl tMSG1392; 

////////////////////////////////////////////////////////////////////////
//
//     Message 1292: Magna Control Input Message
//
typedef tMagnaControl tMSG1292; 

        
////////////////////////////////////////////////////////////////////////
//
//    ZQUERY: Query Message
//
typedef struct {
   tSHORT          MsgID;               // ND  
   tSHORT          MsgFlags;            // connect, disconnect, log control
} tZQUERY;    
     
////////////////////////////////////////////////////////////////////////
//
//    ZLOG: Log Message
//
typedef struct {
   tSHORT          MsgID;               // ND  
   tSHORT          MsgFlags;            // connect, disconnect, log control
   tSHORT          MsgTrigger;          // 0=on time
   tSHORT          MsgOffset;           // secs
   tSHORT          MsgInterval;         // secs
} tZLOG;         

////////////////////////////////////////////////////////////////////////
//
//    ZMSG: structure of structures  
//
typedef struct {  
   tMSG1000 Msg1000;    
   tMSG1001 Msg1001; 
   tMSG1002 Msg1002; 
   tMSG1003 Msg1003;
   tMSG1005 Msg1005;
   tMSG1005 Msg1006;
   tMSG1007 Msg1007;
   tMSG1008 Msg1008;
   tMSG1009 Msg1009;
   tMSG1010 Msg1010;
   tMSG1011 Msg1011;
   tMSG1012 Msg1012;
   tMSG1040 Msg1040;
   tMSG1041 Msg1041;
   tMSG1042 Msg1042;
   tMSG1050 Msg1050;
   tMSG1051 Msg1051;
   tMSG1070 Msg1070;
   tMSG1071 Msg1071;
   tMSG1072 Msg1072;
   tMSG1075 Msg1075;
   tMSG1090 Msg1090;
   tMSG1091 Msg1091;
   tMSG1092 Msg1092;
   tMSG1100 Msg1100;
   tMSG1102 Msg1102;
   tMSG1106 Msg1106;    
   tMSG1108 Msg1108;    
   tMSG1110 Msg1110;
   tMSG1117 Msg1117;
   tMSG1130 Msg1130;
   tMSG1135 Msg1135;
   tMSG1136 Msg1136;
   tMSG1137 Msg1137;
   tMSG1150 Msg1150;
   tMSG1160 Msg1160;
   tMSG1170 Msg1170;
   tMSG1171 Msg1171;
   tMSG1172 Msg1172;
   tMSG1173 Msg1173;
   tMSG1180 Msg1180;
   tMSG1190 Msg1190;
   tMSG1191 Msg1191;
//   tMSG1192 Msg1192;  take out by hphung 12/15/99
   tMSG1200 Msg1200;
   tMSG1208 Msg1208;
   tMSG1210 Msg1210;
   tMSG1211 Msg1211;
   tMSG1212 Msg1212;
   tMSG1213 Msg1213;
   tMSG1214 Msg1214;
   tMSG1216 Msg1216;
   tMSG1217 Msg1217;
   tMSG1218 Msg1218;
   tMSG1219 Msg1219;
   tMSG1220 Msg1220;
   tMSG1221 Msg1221;
   tMSG1240 Msg1240;
   tMSG1241 Msg1241;
   tMSG1242 Msg1242;
   tMSG1270 Msg1270;
   tMSG1292 Msg1292;
   tMSG1300 Msg1300;
   tMSG1303 Msg1303;
   tMSG1304 Msg1304;
   tMSG1305 Msg1305;
   tMSG1306 Msg1306;
   tMSG1310 Msg1310;
   tMSG1317 Msg1317;
   tMSG1330 Msg1330;
   tMSG1331 Msg1331;
   tMSG1334 Msg1334;
   tMSG1337 Msg1337;
   tMSG1360 Msg1360;
   tMSG1380 Msg1380;
   tMSG1390 Msg1390;
//   tMSG1392 Msg1392; take out by hphung 12/15/99
   tZQUERY  ZQuery;
   tZLOG    ZLog;
} tZMSG;  
 
///////////////////////////////////////////////////////////////////////////////
//
//    function prototypes
//
tBOOL DecZMsg (tMSGBUF  *ZMsgBuf, tZMSG ZMsg);
tBOOL BldZMsg (tMSGBUF  *ZMsgBuf, tZMSG ZMsg, tUSHORT MsgNum);

void  Dec1000 (tMSGBUF  *ZMsgBuf, tMSG1000 *Msg1000);
void  Bld1000 (tMSGBUF  *ZMsgBuf, tMSG1000 *Msg1000); 
void  Show1000(tMSG1000 *Msg1000); 
void  Dec1001 (tMSGBUF  *ZMsgBuf, tMSG1001 *Msg1001);
void  Bld1001 (tMSGBUF  *ZMsgBuf, tMSG1001 *Msg1001); 
void  Show1001(tMSG1001 *Msg1001);
void  Dec1002(tMSGBUF   *ZMsgBuf, tMSG1002 *Msg1002);
void  Bld1002(tMSGBUF   *ZMsgBuf, tMSG1002 *Msg1002);
void  Show1002(tMSG1002 *Msg1002);
void  Dec1003(tMSGBUF   *ZMsgBuf, tMSG1003 *Msg1003);
void  Bld1003(tMSGBUF   *ZMsgBuf, tMSG1003 *Msg1003);
void  Show1003(tMSG1003 *Msg1003);
void  Dec1005(tMSGBUF   *ZMsgBuf, tMSG1005 *Msg1005);
void  Bld1005(tMSGBUF   *ZMsgBuf, tMSG1005 *Msg1005);
void  Show1005(tMSG1005 *Msg1005);
void  Dec1007(tMSGBUF   *ZMsgBuf, tMSG1007 *Msg1007);
void  Bld1007(tMSGBUF   *ZMsgBuf, tMSG1007 *Msg1007);
void  Show1007(tMSG1007 *Msg1007);
void  Dec1008(tMSGBUF   *ZMsgBuf, tMSG1008 *Msg1008);
void  Bld1008(tMSGBUF   *ZMsgBuf, tMSG1008 *Msg1008);
void  Show1008(tMSG1008 *Msg1008);
void  Dec1009(tMSGBUF   *ZMsgBuf, tMSG1009 *Msg1009);
void  Bld1009(tMSGBUF   *ZMsgBuf, tMSG1009 *Msg1009);
void  Show1009(tMSG1009 *Msg1009);
void  Dec1011(tMSGBUF   *ZMsgBuf, tMSG1011 *Msg1011);
void  Bld1011(tMSGBUF   *ZMsgBuf, tMSG1011 *Msg1011);
void  Show1011(tMSG1011 *Msg1011);
void  Dec1012(tMSGBUF   *ZMsgBuf, tMSG1012 *Msg1012);
void  Bld1012(tMSGBUF   *ZMsgBuf, tMSG1012 *Msg1012);
void  Show1012(tMSG1012 *Msg1012);
void  Dec1040(tMSGBUF   *ZMsgBuf, tMSG1040 *Msg1040);
//void  Bld1040(tMSGBUF   *ZMsgBuf, tMSG1040 *Msg1040);
//void  Show1040(tMSG1040 *Msg1040);
void  Dec1041(tMSGBUF   *ZMsgBuf, tMSG1041 *Msg1041);
void  Dec1042(tMSGBUF   *ZMsgBuf, tMSG1042 *Msg1042);
void  Dec1050(tMSGBUF   *ZMsgBuf, tMSG1050 *Msg1050);
void  Show1050(tMSG1050 *Msg1050);
void  Dec1051(tMSGBUF   *ZMsgBuf, tMSG1051 *Msg1051);
void  Show1051(tMSG1051 *Msg1051);
void  Dec1070(tMSGBUF   *ZMsgBuf, tMSG1070 *Msg1070);
void  Bld1070(tMSGBUF   *ZMsgBuf, tMSG1070 *Msg1070);
void  Show1070(tMSG1070 *Msg1070);
void  Dec1071(tMSGBUF   *ZMsgBuf, tMSG1071 *Msg1071);
//void  Bld1071(tMSGBUF   *ZMsgBuf, tMSG1071 *Msg1071);
//void  Show1071(tMSG1071 *Msg1071);
void  Dec1072(tMSGBUF   *ZMsgBuf, tMSG1072 *Msg1072);
//void  Bld1072(tMSGBUF   *ZMsgBuf, tMSG1072 *Msg1072);
//void  Show1072(tMSG1072 *Msg1072);

void  Dec1075(tMSGBUF   *ZMsgBuf, tMSG1075 *Msg1075);
void  Bld1075(tMSGBUF   *ZMsgBuf, tMSG1075 *Msg1075);
void  Show1075(tMSG1075 *Msg1075);
void  Dec1090(tMSGBUF   *ZMsgBuf, tMSG1090 *Msg1090);
void  Bld1090(tMSGBUF   *ZMsgBuf, tMSG1090 *Msg1090);
void  Show1090(tMSG1090 *Msg1090);
void  Dec1091(tMSGBUF   *ZMsgBuf, tMSG1091 *Msg1091);
void  Bld1091(tMSGBUF   *ZMsgBuf, tMSG1091 *Msg1091);
void  Show1091(tMSG1091 *Msg1091);
void  Dec1092(tMSGBUF   *ZMsgBuf, tMSG1092 *Msg1092);
void  Show1092(tMSG1092 *Msg1092);  
     
void  Dec1100(tMSGBUF   *ZMsgBuf, tMSG1100 *Msg1100);
void  Bld1100(tMSGBUF   *ZMsgBuf, tMSG1100 *Msg1100);
void  Show1100(tMSG1100 *Msg1100);  
void  Show1102(tMSG1102 *Msg1102);
void  Dec1106(tMSGBUF   *ZMsgBuf, tMSG1106 *Msg1106);
void  Show1106(tMSG1106 *Msg1106);
void  Show1108(tMSG1108 *Msg1108); 
void  Dec1110(tMSGBUF   *ZMsgBuf, tMSG1110 *Msg1110);
void  Bld1110(tMSGBUF   *ZMsgBuf, tMSG1110 *Msg1110);
void  Show1110(tMSG1110 *Msg1110);  
void  Dec1117(tMSGBUF   *ZMsgBuf, tMSG1117 *Msg1117);
void  Bld1117(tMSGBUF   *ZMsgBuf, tMSG1117 *Msg1117);
void  Show1117(tMSG1117 *Msg1117);  
void  Dec1130(tMSGBUF   *ZMsgBuf, tMSG1130 *Msg1130);
void  Bld1130(tMSGBUF   *ZMsgBuf, tMSG1130 *Msg1130);
void  Show1130(tMSG1130 *Msg1130);  
void  Dec1135(tMSGBUF   *ZMsgBuf, tMSG1135 *Msg1135);
void  Bld1135(tMSGBUF   *ZMsgBuf, tMSG1135 *Msg1135);
void  Show1135(tMSG1135 *Msg1135);  
void  Dec1136(tMSGBUF   *ZMsgBuf, tMSG1136 *Msg1136);
void  Bld1136(tMSGBUF   *ZMsgBuf, tMSG1136 *Msg1136);
void  Show1136(tMSG1136 *Msg1136);  
void  Dec1137(tMSGBUF   *ZMsgBuf, tMSG1137 *Msg1137);
//void  Bld1137(tMSGBUF   *ZMsgBuf, tMSG1137 *Msg1137);
void  Show1137(tMSG1137 *Msg1137);  
//void  Dec1138(tMSGBUF   *ZMsgBuf, tMSG1138 *Msg1138);
//void  Bld1138(tMSGBUF   *ZMsgBuf, tMSG1138 *Msg1138);
//void  Show1138(tMSG1138 *Msg1138);  
//void  Dec1150(tMSGBUF   *ZMsgBuf, tMSG1150 *Msg1150);
//void  Bld1150(tMSGBUF   *ZMsgBuf, tMSG1150 *Msg1150);
//void  Show1150(tMSG1150 *Msg1150);  
void  Dec1160(tMSGBUF   *ZMsgBuf, tMSG1160 *Msg1160);
void  Bld1160(tMSGBUF   *ZMsgBuf, tMSG1160 *Msg1160);
void  Show1160(tMSG1160 *Msg1160);  
void  Dec1170(tMSGBUF   *ZMsgBuf, tMSG1170 *Msg1170);
void  Bld1170(tMSGBUF   *ZMsgBuf, tMSG1170 *Msg1170);
void  Show1170(tMSG1170 *Msg1170);  
void  Dec1171(tMSGBUF   *ZMsgBuf, tMSG1171 *Msg1171);
void  Bld1171(tMSGBUF   *ZMsgBuf, tMSG1171 *Msg1171);
void  Show1171(tMSG1171 *Msg1171);  
void  Dec1172(tMSGBUF   *ZMsgBuf, tMSG1172 *Msg1172);
void  Bld1172(tMSGBUF   *ZMsgBuf, tMSG1172 *Msg1172);
void  Show1172(tMSG1172 *Msg1172);  
void  Dec1173(tMSGBUF   *ZMsgBuf, tMSG1173 *Msg1173);
void  Bld1173(tMSGBUF   *ZMsgBuf, tMSG1173 *Msg1173);
void  Show1173(tMSG1173 *Msg1173);  
void  Dec1180(tMSGBUF   *ZMsgBuf, tMSG1180 *Msg1180);
void  Bld1180(tMSGBUF   *ZMsgBuf, tMSG1180 *Msg1180);
void  Show1180(tMSG1180 *Msg1180);  
void  Dec1190(tMSGBUF   *ZMsgBuf, tMSG1190 *Msg1190);
void  Bld1190(tMSGBUF   *ZMsgBuf, tMSG1190 *Msg1190);
void  Show1190(tMSG1190 *Msg1190);  
void  Dec1191(tMSGBUF   *ZMsgBuf, tMSG1191 *Msg1191);
void  Bld1191(tMSGBUF   *ZMsgBuf, tMSG1191 *Msg1191);
void  Show1191(tMSG1191 *Msg1191);  
//void  Dec1192(tMSGBUF   *ZMsgBuf, tMSG1192 *Msg1192);    take out because of msgID changed
//void  Show1192(tMSG1192 *Msg1192);                       1192 --> 1092

void  Dec1200(tMSGBUF   *ZMsgBuf, tMSG1200 *Msg1200);
void  Bld1200(tMSGBUF   *ZMsgBuf, tMSG1200 *Msg1200);
void  Dec1208(tMSGBUF   *ZMsgBuf, tMSG1208 *Msg1208);
void  Bld1208(tMSGBUF   *ZMsgBuf, tMSG1208 *Msg1208);
void  Dec1210(tMSGBUF   *ZMsgBuf, tMSG1210 *Msg1210);
void  Bld1210(tMSGBUF   *ZMsgBuf, tMSG1210 *Msg1210); 
void  Dec1211(tMSGBUF   *ZMsgBuf, tMSG1211 *Msg1211);
void  Bld1211(tMSGBUF   *ZMsgBuf, tMSG1211 *Msg1211); 
void  Dec1212(tMSGBUF   *ZMsgBuf, tMSG1212 *Msg1212);
void  Bld1212(tMSGBUF   *ZMsgBuf, tMSG1212 *Msg1212); 
void  Dec1213(tMSGBUF   *ZMsgBuf, tMSG1213 *Msg1213);
void  Bld1213(tMSGBUF   *ZMsgBuf, tMSG1213 *Msg1213); 
void  Dec1214(tMSGBUF   *ZMsgBuf, tMSG1214 *Msg1214);
void  Bld1214(tMSGBUF   *ZMsgBuf, tMSG1214 *Msg1214);
void  Dec1216(tMSGBUF   *ZMsgBuf, tMSG1216 *Msg1216);
void  Bld1216(tMSGBUF   *ZMsgBuf, tMSG1216 *Msg1216); 
void  Dec1217(tMSGBUF   *ZMsgBuf, tMSG1217 *Msg1217);
void  Bld1217(tMSGBUF   *ZMsgBuf, tMSG1217 *Msg1217); 
//void  Dec1218(tMSGBUF   *ZMsgBuf, tMSG1218 *Msg1218);
void  Bld1218(tMSGBUF   *ZMsgBuf, tMSG1218 *Msg1218); 
void  Dec1219(tMSGBUF   *ZMsgBuf, tMSG1219 *Msg1219);
void  Bld1219(tMSGBUF   *ZMsgBuf, tMSG1219 *Msg1219); 
void  Dec1220(tMSGBUF   *ZMsgBuf, tMSG1220 *Msg1220);
void  Bld1220(tMSGBUF   *ZMsgBuf, tMSG1220 *Msg1220); 
void  Dec1221(tMSGBUF   *ZMsgBuf, tMSG1221 *Msg1221);
void  Bld1221(tMSGBUF   *ZMsgBuf, tMSG1221 *Msg1221);
//void  Dec1240(tMSGBUF   *ZMsgBuf, tMSG1240 *Msg1240);
void  Bld1240(tMSGBUF   *ZMsgBuf, tMSG1240 *Msg1240);
void  Bld1241(tMSGBUF   *ZMsgBuf, tMSG1241 *Msg1241);
void  Bld1242(tMSGBUF   *ZMsgBuf, tMSG1242 *Msg1242);
//void  Dec1270(tMSGBUF   *ZMsgBuf, tMSG1270 *Msg1270);
void  Bld1270(tMSGBUF   *ZMsgBuf, tMSG1270 *Msg1270);
void  Bld1292(tMSGBUF   *ZMsgBuf, tMSG1292 *Msg1292);

void  Dec1300(tMSGBUF   *ZMsgBuf, tMSG1300 *Msg1300);
void  Bld1300(tMSGBUF   *ZMsgBuf, tMSG1300 *Msg1300);
void  Dec1303(tMSGBUF   *ZMsgBuf, tMSG1303 *Msg1303);
void  Bld1303(tMSGBUF   *ZMsgBuf, tMSG1303 *Msg1303); 
//void  Dec1304(tMSGBUF   *ZMsgBuf, tMSG1304 *Msg1304);
void  Bld1304(tMSGBUF   *ZMsgBuf, tMSG1304 *Msg1304); 
void  Bld1305(tMSGBUF   *ZMsgBuf, tMSG1305 *Msg1305); 
void  Bld1306(tMSGBUF   *ZMsgBuf, tMSG1306 *Msg1306); 
void  Dec1310(tMSGBUF   *ZMsgBuf, tMSG1310 *Msg1310);
void  Bld1310(tMSGBUF   *ZMsgBuf, tMSG1310 *Msg1310);
void  Dec1317(tMSGBUF   *ZMsgBuf, tMSG1317 *Msg1317);
void  Bld1317(tMSGBUF   *ZMsgBuf, tMSG1317 *Msg1317);
void  Dec1330(tMSGBUF   *ZMsgBuf, tMSG1330 *Msg1330);
void  Bld1330(tMSGBUF   *ZMsgBuf, tMSG1330 *Msg1330); 
void  Dec1331(tMSGBUF   *ZMsgBuf, tMSG1331 *Msg1331);
void  Bld1331(tMSGBUF   *ZMsgBuf, tMSG1331 *Msg1331); 
void  Bld1334(tMSGBUF   *ZMsgBuf, tMSG1334 *Msg1334); 
void  Bld1337(tMSGBUF   *ZMsgBuf, tMSG1337 *Msg1337); 
//void  Dec1350(tMSGBUF   *ZMsgBuf, tMSG1350 *Msg1350);
//void  Bld1350(tMSGBUF   *ZMsgBuf, tMSG1350 *Msg1350);
//void  Dec1360(tMSGBUF   *ZMsgBuf, tMSG1360 *Msg1360);
void  Bld1360(tMSGBUF   *ZMsgBuf, tMSG1360 *Msg1360);
//void  Dec1380(tMSGBUF   *ZMsgBuf, tMSG1380 *Msg1380);
void  Bld1380(tMSGBUF   *ZMsgBuf, tMSG1380 *Msg1380);
void  Bld1390(tMSGBUF   *ZMsgBuf, tMSG1390 *Msg1390);
//void  Bld1392(tMSGBUF   *ZMsgBuf, tMSG1392 *Msg1392); take out because of msgID changed
                                                      //1392 --> 1292
void  DecZLog(tMSGBUF   *ZMsgBuf, tZLOG    *ZLog);
void  BldZLog(tMSGBUF   *ZMsgBuf, tZLOG    *ZLog);
void  DecZQuery(tMSGBUF *ZMsgBuf, tZQUERY  *ZQuery);
void  BldZQuery(tMSGBUF *ZMsgBuf, tZQUERY  *ZQuery);

// Zodiac message super-structure
//
extern tZMSG  ZMsg;

// Unpacked versions of Zodiac message (sub)structures
//
extern tDIFFSTATUSBITS     DiffStatusBits;                 // diff status
extern tCORRSTATUSBITS     CorrStatusBits[GI_MAX_VIS_SVS]; // corr status
extern tSOLUTIONSTATUSBITS SolutionStatusBits;             // sol validity stat
extern tCHNSUMDATA         ChnSumData[GI_NUM_HW_CHAN];     // chan summary
extern tINITVALIDDATA      InitValidData;                  // init validity
extern tINITMODEDATA       InitModeData;                   // init timemark mode
extern tCOMMANDWORD        CommandWord;                    // command word
extern tSOLVALIDDATA       SolValidData;                   // sol validity data
extern tNAVCFGBITS         NavCfgBits;                     // nav config
extern tCONFIGDATA         ConfigData;                     // config data
extern tBOOL               CandidatePRN[32];               // candidate SV PRNs
extern tSPVALIDDATA        SPValidData;                    // serial port cfg
extern tANTENNADATA        AntennaData;                    // antenna
extern tDGPSCONTROLDATA    DGPSControlData;                // DGPS control
extern tCOLDSTARTDATA      ColdStartData;                  // cold start
extern tUSERALTINPUTDATA   AltInputData;                   // altitude input
extern tBOOL               DeviceNotPresent;               // device status
extern tBOOL               vEEPROMStatus;                  // vEEPROM checksum status
extern tEEPROMUPDATEDATA   EEPROMUpdateData;               // EEPROM update
extern tFRQTBLDATA         FrqTblData[256];                // freq table
extern tBITS1170           Bits1170;                       // bitfield in 1170
extern tBITS1171           Bits1171;                       // bitfield in 1171
extern tBESTCHANSUM        BestChanSum[GI_NUM_HW_CHAN];    // Best Channel Data for DR


